/*
 *
 * Linux-x86 software for eprommer "Quickbyte 2.0 (Amiga)"
 * ========================================================================
 * Copyright 1996 thh-computergraphics. Refer to GPL for copying.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this software; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * Thorsten Hempel
 * Meisterstr. 23
 * 23919 Berkenthin
 * thh@valis.pumuckl.cubenet.de
 *
 *
 * prommer v1.0
 * Das menuegefuehrte Steuerprogramm zur Epromprogrammierung.
 * Sobald ein paar weitere Tests erfolgreich verlaufen, wird
 * es ein groesseres cleanup geben.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <assert.h>

#include <unistd.h>
#include <asm/io.h>
#include <linux/delay.h>	

#include "tio.h"
#include "prommer-hw.h"



#ifndef FALSE
#define FALSE 0
#endif
#ifndef TRUE
#define TRUE (!FALSE)
#endif

typedef int bool;
typedef	unsigned char ubyte;

int debugflag=FALSE;		/* erzeugt viel muell an stderr... */

/*----------------------------------------------------------*/
/* Low level Interface                                      */
/* Wenn zur Datenausgabe NUR WriteCommand verwendet wird,   */
/* kann durch das Flag U3OE der Zustand von OE von U3       */
/* einfach und zuverlaessig verwaltet werden.               */
/* (/dev/tio setzt OE von U3 sowieso immer)                 */
/*----------------------------------------------------------*/

int tio_id;			/* descriptor fuer /dev/tio */

#define WriteCommand(cmd) outb(cmd|U3OE,LPTBASE)
#define GetBit() ((inb(LPTSTATUS) & BUSYMASK)!=0)


/*----------------------------------------------------------*/
/* Definition der Optionen                                  */
/*----------------------------------------------------------*/

typedef struct {
	ulong	type;			/* 2716..27512 */
	ulong	size;			/* 2KB..64KB */
	char	type_name[32];		/* "2716".."27512" */
	float	voltage;		/* 5.0/12.5/21.0/25.0 */
	char	voltage_name[32];	/* "5.0"/"12.5"/"21.0"/"25.0" */
	int	algorithm;		/* 1=50ms 2=quick1 3=quick2 4=cust1 5=cust2 */
	char	algorithm_name[64];	/* "normal", "quick1", "quick2" "c1" "c2" */
	long	port;			/* LPTx adresse */
	char	port_name[32];		/* "LPTx" */
	/* datei und brennbereich */
	char	filename[1024];		/* "image.rom" */
	long	start;			/* 0 */
	long	ende;			/* size -1 */
	long	offset;			/* 0 */
	/* werte fuer konfigurierbare custom routinen */
	long	c_delay;		/* impulsdauer */
	long	c_min;			/* minimale inpulszahl */
	long	c_max;			/* maximale impulszahl */
	long	c_faktor;		/* anzahl nachbrennimpulse */
	long	c_sdelay;		/* impulsdauer */
	long	c_snum;			/* impulszahl */
	} options;

/* nur eine instanz noetig */
options Optionen;


#define LPTBASE (Optionen.port)		/* keine konstanten! */
#define LPTDATA LPTBASE
#define LPTSTATUS (LPTBASE+1)
#define LPTSTEUER (LPTBASE+2)


/*----------------------------------------------------------*/
/* basic io structures/modellierung der hardware            */
/*----------------------------------------------------------*/

/* modelliert einen cd4094 */
typedef struct {
	int	clock;		/* Ausgabebit dieses Chips fuer DB-25 */
	bool	out[8];
	} cd4094;

#define U3OE_ON  0x01
#define U3OE_OFF 0x00

/* die chips/pins */
cd4094	u3,u4,u5;		/* 3 Instanzen vorhanden */
bool	U3OE=U3OE_OFF;		/* OE von U3 (4094) setzen */
long	act;			/* aktuelle adresse am eprom */


/*----------------------------------------------------------*/
/* cd4094 interface                                         */
/*----------------------------------------------------------*/

/* etwa 26 zugriffe pro byte */

/* schreibt den inhalt der 4094 structure an den realen chip */
void cd4094Send(cd4094 *device){
int i;
int clock;

	assert(device!=NULL);

	clock=device->clock;
	for (i=7;i>=0;i--) {
		/* data fuer u3/u4/u5 ist bit D7 (128) */
		if (device->out[i]) {
			WriteCommand(BIT_DATA345);
			/* clock hi erzeugen (hier erfolgt datenuebernahme) */
			WriteCommand(clock| BIT_DATA345);
			/* clock lo erzeugen */
/*			WriteCommand(BIT_DATA345); (kann entfallen) */
			/* clock lo standard */
			WriteCommand(0);
			}
		else {
			/* clock hi erzeugen */
			WriteCommand(clock);
			/* clock lo erzeugen */
			WriteCommand(0);
			}
		}
	/* strobe = bit D6 (64) */
	/* strobe hi erzeugen */
	WriteCommand(BIT_STROBE345);
	/*  strobe lo erzeugen */
	WriteCommand(0);
}


void cd4094Init(int clock, cd4094 *device){
int i;
	assert(device!=NULL);		
	for (i=7;i>=0;i--) {
		device->out[i]=0;
		}
	device->clock=clock;
	cd4094Send(device);
}


/* schreibt value in die 4094 structure und an den realen chip */
void cd4094Write(cd4094 *device, ubyte value){
int i;
	assert(device!=NULL);
	for (i=0;i<8;i++) {
		device->out[i]= ((value&1)!=0) ;
		value=value>>1;
		}
	cd4094Send(device);
}	


/* liest value von 4094 structure */
ubyte cd4094Read(cd4094 *device){
ubyte value=0;
int i;
	assert(device!=NULL);

	for (i=0;i<8;i++) {
		value=value<<1;
		if (device->out[7-i]) value++;
		}
	return(value);
}	


/*----------------------------------------------------------*/
/* Byte vom prommer lesen (4021)                            */
/*----------------------------------------------------------*/

/* reset erfolgt durch jeden strobe auf einem der 4094 (u3..u5) */
/* read actual byte from EPROM with 4021 */
int ReadByte4021(){
long i;
int result;

	WriteCommand(0);
	/* reset 4021: strobe auf D6 "64" */
	WriteCommand(BIT_STROBE345);
	WriteCommand(0);

	result=0;
	for (i=0;i<8;i++) {
		result=result<<1;
		if (GetBit()) result=result|1;
		/* clock strobe auf CLOCK fuer U2 (4021) erzeugen */
		WriteCommand(BIT_CLOCK2);
		WriteCommand(0);

		}
	return(result);
}
		

/* liest aktuelles byte vom eprom ein, erzeugt dazu die noetigen steuersignale */
int ReadByteEprom(){
int value;
	u4.out[0]=0;		/* CE -> 0 */
	u4.out[1]=0;		/* OE -> 0 */
	cd4094Send(&u4);	/* OE und CE schreiben == Reset fuer 4021 */
	value=ReadByte4021();	/* von 4021 */
	u4.out[0]=1;		/* CE -> 1 */
	u4.out[1]=1;		/* OE -> 1 */
	cd4094Send(&u4);	/* OE und CE schreiben */
	return(value);
}



/*----------------------------------------------------------*/
/* Interface zum 4040                                       */
/*----------------------------------------------------------*/

/* reset counter 4040 (U1): 0->1->0 */
void Reset4040(){
	u5.out[2]=1;
	cd4094Send(&u5);
	u5.out[2]=0;
	cd4094Send(&u5);
	act=act&(1+2+4+8+16+32+64+128+256+512);	/* aktuelle Adresse */
}


/*----------------------------------------------------------*/
/* Spannung auswaehlen			                    */
/*----------------------------------------------------------*/

int SelectVoltage(float voltage) {
int v;
	v=voltage*10.0;
	u5.out[5]=0;		
	u5.out[6]=0;
	u5.out[7]=0;
	if (v==50) {
		/* 5.0 V waehlen */
		return(TRUE);
		}
	else if (v==125) {
		/* 12.5 V waehlen */
		u5.out[5]=1;		
		return(TRUE);
		}
	else if (v==210) {
		/* 21.0 V waehlen */
		u5.out[6]=1;		
		return(TRUE);
		}
	else if (v==250) {
		/* 25.0 V waehlen */
		u5.out[7]=1;		
		return(TRUE);
		}
	else {
		fprintf(stderr,"illegale spannung: %g (%d)\n",voltage,v);
		return(FALSE);
		}
}


/*----------------------------------------------------------*/
/* LED interface                                            */
/*----------------------------------------------------------*/

/* LED schalten */
void LED(bool onoff) {
	u5.out[1]=onoff;	/* LED-Bit */
	cd4094Send(&u5);	/* LED schreiben */
}



/*----------------------------------------------------------*/
/* Adressauswahl (universell)                               */
/*----------------------------------------------------------*/

/* wird noch nicht verwendet, da in dieser form nicht brauchbar! */
void Addr(long address){
long i;
int command;
volatile long high,low10;
volatile long act10,acthigh;

	/* -1 als aktuelle adresse besagt, dass die auf jeden */
	/* fall neu gebildet werden muss! */
	if (act==-1) Reset4040();

/*	if (address==act) return;*/


	command=0;WriteCommand(command);

	/* adressbits trennen */	
	low10=address & (1+2+4+8+16+32+64+128+256+512);
	act10=act & (1+2+4+8+16+32+64+128+256+512);
	if (act10>low10) Reset4040();
	/* act10 neu berechnen */
	act10=act & (1+2+4+8+16+32+64+128+256+512);

	/* low part of address */
	for (i=act10;i<low10;i++) {
		/* clock strobe auf CLOCK fuer U1 (4040) erzeugen */
		WriteCommand(BIT_CLOCK1);
		WriteCommand(0);
		}

	high=address>>10;	
	acthigh=act>>10;	

	/* high part of address in U4 (4094) */
	u4.out[3]=(high & 1); 		/* bit 10 */
	u4.out[4]=(high & 2); 		/* bit 11 */
	u4.out[5]=(high & 4); 		/* bit 12 */
	u4.out[6]=!(high & 8); 		/* bit 13 ab 27128 (INVERS!) */
	u4.out[7]=(high & 16); 		/* bit 14 ab 27256 */
	if (Optionen.type==27512) {
		u5.out[0]=(high & 32); 	/* bit 15 ab 27512; u4[2] muss 0 sein! */
		}
	else {
		u5.out[0]=1; 		/* "Vpp" bei 27256 (hier: +5V) */
		}

	cd4094Send(&u4);
	cd4094Send(&u5);
	act=address;
}

/* stattdessen werden diese spezialfunktionen verwendet */
void Addr2716(long address);
void Addr2764(long address);
void Addr27256(long address);


/*----------------------------------------------------------*/
/* 2764/27128 Routinen (sollten funktionieren)              */
/*----------------------------------------------------------*/


/* vorbereitung zum einsezten des eproms */
void Init2764(){

	Reset4040();

	/* ausgaberegister egal */
	cd4094Write(&u3,0);	
	/* Steuerregister 4:  OE:=1, CE:=1, PGM==A14:=1 */
	cd4094Write(&u4,128+1+2);	

	/* Steuerregister 5:  Vpp an Pin 1 auf 5V, LED aus */
	/*SelectVoltage(5.0);*/
	cd4094Write(&u5,16);	
	Addr2764(0);
	LED(FALSE);
}



void Done2764(){

	/* ausgaberegister egal */
	cd4094Write(&u3,0);	

	/* Steuerregister 4:  OE:=1, CE:=1, PGM==A14:=1 */
	cd4094Write(&u4,128+1+2);	

	/* Steuerregister 5:  Vpp an Pin 1 auf 5V, LED aus */
	cd4094Write(&u5,16);	
	Addr2764(0);
	LED(FALSE);
}



void Addr2764(long address){
long i;
int command;
volatile long high,low10;
volatile long act10,acthigh;

	/* -1 als aktuelle adresse besagt, dass die auf jeden */
	/* fall neu gebildet werden muss! */
	if (act==-1) Reset4040();

/*	if (address==act) return;*/

	command=0;WriteCommand(command);

	/* adressbits trennen */	
	low10=address & (1+2+4+8+16+32+64+128+256+512);
	act10=act & (1+2+4+8+16+32+64+128+256+512);
	if (act10>low10) Reset4040();
	/* act10 neu berechnen */
	act10=act & (1+2+4+8+16+32+64+128+256+512);

	/* low part of address */
	for (i=act10;i<low10;i++) {
		/* clock strobe auf CLOCK fuer U1 (4040) erzeugen */
		WriteCommand(BIT_CLOCK1);
		WriteCommand(0);
		}

	high=address>>10;	
	acthigh=act>>10;	
	/* high part of address in U4 (4094) */

	u4.out[3]=(high & 1); 		/* bit 10 */
	u4.out[4]=(high & 2); 		/* bit 11 */
	u4.out[5]=(high & 4); 		/* bit 12 */
	u4.out[6]=!(high & 8); 		/* bit 13 (ab 27128) (INVERS!) */

	cd4094Send(&u4);
	act=address;
}


void Read2764(long start, long ende, char *filename){
char s[80];
long i;
int value;
FILE *f;

	assert(filename!=NULL);

	printf("Auslesen eines 2764/27128.\n");
	f=fopen(filename,"w");if (!f) return;

	Init2764();
	printf("Bitte Eprom 2764/27128 einsetzen und ENTER druecken");gets(s);

	/* LED einschalten */
	LED(TRUE);

	/* alle adressen auslesen */
	for (i=start;i<=ende;i++){
		Addr2764(i);
		value=ReadByteEprom();	/* von 4021 */
		if (!(i%256)) {printf("\r$%06lx (%06ld): %02x",i,i,value);fflush(stdout);}
		putc(value,f);
		}
	
	/* fertig */
	Done2764();
	printf("\nBitte Eprom 2764/27128 entnehmen und ENTER druecken");gets(s);
	fclose(f);
}



int Check2764(long start, long ende){
char s[80];
long i;
int value;
int result;
FILE *f;

	printf("Test, ob 2764/27128 leer ist.\n");
	result=TRUE;

	Init2764();
	printf("Bitte Eprom 2764/27128 einsetzen und ENTER druecken");gets(s);

	/* LED einschalten */
	LED(TRUE);

	/* alle adressen testen */
	for (i=start;i<=ende;i++){
		Addr2764(i);
		value=ReadByteEprom();	/* von 4021 */
		if (!(i%256)) {printf("\r$%06lx (%06ld): %02x",i,i,value);fflush(stdout);}
		if (value!=255) {
			result=FALSE;
			printf ("%08d: %03d\n",i,value);
			}
		}
	
	/* fertig */
	Done2764();
	printf("\nBitte Eprom 2764/27128 entnehmen und ENTER druecken");gets(s);

	if (result) {
		printf("Eprom 2764/27128 ist leer.\n");
		}
	else {
		printf("Eprom 2764/27128 ist NICHT leer!\n");
		}
	return(result);
}



/* vergleich file-eprom */
int Compare2764(long start, long ende, long offset, char *filename){
char s[80];
long i;
int value_r,value_w;
FILE *f;
int result;

	assert(filename!=NULL);

	printf("Vergleichen eines 2764/27128.\n");
	f=fopen(filename,"r");if (!f) return(FALSE);
	fseek(f,offset,SEEK_SET);
	result=TRUE;

	Init2764();
	printf("Bitte Eprom 2764/27128 einsetzen und ENTER druecken");gets(s);

	/* LED einschalten */
	LED(TRUE);

	/* alle adressen vergleichen */
	for (i=start;i<=ende;i++){
		value_w=fgetc(f);		/* byte lesen */
		Addr2764(i);
		value_r=ReadByteEprom();
		if (!(i%256)) {printf("\r$%06lx (%06ld): %02x",i,i,value_r);fflush(stdout);}
		if (value_w!=value_r) {
			printf ("difference at %08d [%06x]: read %03d [%02x] instead of %03d [%02x]\n",i,i,value_r,value_r,value_w,value_w);
			result=FALSE;
			}
		}
	
	/* fertig */
	Done2764();
	printf("\nBitte Eprom 2764/27128 entnehmen und ENTER druecken");gets(s);
	fclose(f);
	return(result);
}



/* standard-algorithmus, aber $FF werden uebersprungen */
void Write2764(long start, long ende, long offset, char *filename, int num_impulse, long impuls_breite){
char s[80];
long i,j;
int value_r,value_w;
FILE *f;
tiostruct parms;

	assert(filename!=NULL);
	printf("Brennen eines 2764/27128.\n");

	f=fopen(filename,"r");if (!f) return;
	fseek(f,offset,SEEK_SET);

	Init2764();
	printf("Bitte Eprom 2764/27128 einsetzen und ENTER druecken");gets(s);

	/* LED einschalten */
	LED(TRUE);

/* programmiermodus einschalten */		

	/* Programmierspannung xxV an Eprom Pin 1 einstellen */
	u5.out[0]=1;		/* Pin1 von Eprom setzen */
	SelectVoltage(Optionen.voltage);	
	cd4094Send(&u5);	/* U5 schreiben, setzt auch OE von U3 */
	usleep(1000000);	/* 1 sekunde warten */

	/* port setzen (einmal reicht) */
	parms.port=LPTBASE;
	parms.delay=impuls_breite-us_error;

	printf("\nprogramming:\n");

	/* alle adressen schreiben */
	for (i=start;i<=ende;i++){
		if (!(i%16)) {printf("\r$%06lx (%ld)",i,i);fflush(stdout);}
		value_w=fgetc(f);		/* byte von file lesen */
		Addr2764(i);			/* adresse an 2764 einstellen */

		/* wert -> 4094 */
		cd4094Write(&u3,value_w);
		U3OE=U3OE_ON;		/* OE von U5 (4094) enable bei naechstem Kommando! */
		
/* programmieren (aehnlich lesen, aber OE vom Eprom bleibt auf 1!) */

		u4.out[0]=0;			/* CE -> 0 */
		cd4094Send(&u4);		/* CE schreiben, setzt auch OE von U3 */

		parms.wert1=cd4094Read(&u4) & 127;	/* PGM -> 0 */
		parms.wert2=cd4094Read(&u4) | 128;	/* PGM -> 1 */

		/* innere "brennschleife" */
		if (value_w!=255) for (j=0;j<num_impulse;j++) {
			/* schreibt werte[0], wartet 1000us, schreibt werte[1] */
			if (ioctl(tio_id,TIO_PROG,&parms));
			} /* for j */

		u4.out[0]=1;			/* CE -> 1 */
		cd4094Send(&u4);		/* CE schreiben */
		U3OE=U3OE_OFF;		/* OE von U5 (4094) disable bei naechstem Kommando! */
		} /* for (alle zu brennenden adressen) */

/* fertig */
	printf("\n");

/* programmiermodus abschalten */
		
	/* Programmierspannung 25V an Eprom Pin 1 abstellen */
	u5.out[0]=0;		/* Pin1 von Eprom setzen */
	SelectVoltage(5.0);	/* 5V ist "normale" spannung */
	cd4094Send(&u5);	/* U5 schreiben, setzt auch OE von U3 */
	usleep(1000000);	/* 1 sekunde warten */

	/* fertig */
	Done2764();
	printf("\nBitte Eprom 2764/27128 entnehmen und ENTER druecken");gets(s);
	fclose(f);
}



/* quick 1 auf bitebene, Vpp wird zum lesen jeweils deaktiviert! */
/* hmm, ein ganz klein wenig unuebersichtlich sind diese bitfummeleien in C ja schon... */
void FastWrite2764(long start,long ende,long offset,char *filename,
	int min_impulse,int max_impulse, int faktor, long impuls_breite){
char s[80];
long i,j;
int value_r,value_w;
FILE *f;
bool	bits_in[8],bits_out[8];
int	dauer[8];
int	nachbrennen[8];
int	minimum,maximum,anzahl,bit,r_alt,r_neu;
unsigned int brennbyte;
bool	fehler;
long summe;
tiostruct parms;

	assert(filename!=NULL);
	printf("Brennen eines 2764/27128 mit Quick 1.\n");

	f=fopen(filename,"r");if (!f) return;
	fseek(f,offset,SEEK_SET);

	for (i=0;i<8;i++) {
		bits_in[i]=FALSE;
		bits_out[i]=FALSE;
		dauer[i]=0;
		nachbrennen[i]=0;
		}
	maximum=0;summe=0;minimum=99;anzahl=0;

	Init2764();
	printf("Bitte Eprom 2764/27128 einsetzen und ENTER druecken");gets(s);

	/* LED einschalten */
	LED(TRUE);

	/* Programmierspannung xxV  waehlen */
	SelectVoltage(Optionen.voltage);	
	cd4094Send(&u5);	/* U5 schreiben */
	usleep(1000000);	/* 1 sekunde warten */

	/* port setzen (einmal reicht) */
	parms.port=LPTBASE;
	parms.delay=impuls_breite-us_error;

	printf("\nprogramming:\n");
	fehler=FALSE;

	/* alle adressen schreiben */
	for (i=start;(i<=ende)&&!fehler;i++){
		value_w=fgetc(f);		/* byte von file lesen */
		Addr2764(i);			/* adresse an 2764 einstellen */

		/* 0. testen, ob das byte ueberhaupt programmierbar ist */
		value_r=ReadByteEprom(); 

		if (value_r!=255) {
			fehler=TRUE;
			printf("\nFehler: Byte %d ist nicht $FF sondern $%02x!\n",i,value_r);
			}

		/* 1. feststellen, welche bits von value_w auf 0 zu bringen sind */
		for (bit=7;bit>=0;bit--) {
			if (value_w&(1<<bit)) {
				dauer[bit]=0;
				}
			else {
				dauer[bit]=1;
				anzahl++;		/* 0-Bits zaehlen */
				}
			nachbrennen[bit]=0;
			}

		/* 2. das aktuelle byte an aktueller adresse brennen */
		do {
			/* aktuelles byte aus dem eprom lesen */
			r_alt=ReadByteEprom();
if (debugflag) fprintf(stderr,"Gelesen1[$%04x]: $%02x\n",i,r_alt);
			/* zu brennendes byte erstellen */
			brennbyte=0;
			for (bit=7;bit>=0;bit--) {
				brennbyte=brennbyte<<1;	
				if (nachbrennen[bit]) nachbrennen[bit]--;
				if (dauer[bit]) dauer[bit]++;
				if (dauer[bit] || nachbrennen[bit]) brennbyte++;
				if ((dauer[bit])>max_impulse) {
					printf("\nFehler: $%02x statt $%02x (bit %d)\n",
						r_alt,value_w,bit);
					fehler=TRUE;
					}
				}
			brennbyte=(~brennbyte)&255;	/* gesetzte bits auf 0 brennen */

			/* das byte brennen, $FF immer ueberspringen */
			if (brennbyte!=255) {
				/* wert -> 4094 */
				cd4094Write(&u3,brennbyte);
				U3OE=U3OE_ON;	/* OE von U5 (4094) enable bei next cmd */
				u4.out[0]=0;			/* CE -> 0 */
				cd4094Send(&u4);		/* CE schreiben, setzt auch U3OE*/

				parms.wert1=cd4094Read(&u4) & 127;	/* PGM -> 0 */
				parms.wert2=cd4094Read(&u4) | 128;	/* PGM -> 1 */

				u5.out[0]=1;		/* Pin1 von Eprom auf Vpp setzen */
				cd4094Send(&u5);	/* U5 schreiben */

				/* schreibt wert1, wartet n us, schreibt wert2 */
				if (ioctl(tio_id,TIO_PROG,&parms));

				u5.out[0]=0;		/* Pin1 von Eprom auf 5V setzen */
				cd4094Send(&u5);	/* U5 schreiben */

if (debugflag) fprintf(stderr,"Gebrannt[$%04x]: $%02x\n",i,brennbyte);
				u4.out[0]=1;			/* CE -> 1 */
				cd4094Send(&u4);		/* CE schreiben */
				U3OE=U3OE_OFF;	/* OE von U5 (4094) disable bei next cmd */
				} /* if brennbyte */

			/* ergebnis pruefen */
			r_neu=ReadByteEprom();
if (debugflag) fprintf(stderr,"Gelesen2[$%04x]: $%02x\n",i,r_neu);
			for (bit=7;bit>=0;bit--) {
				int r_exor=(r_alt ^ r_neu);
				if ((!nachbrennen[bit]) && (r_exor&(1<<bit))) {
					/* ja, dieses bit ist gerade jetzt gekippt! */
					dauer[bit]--;	/* dauer ist immer 1 zu gross! */
					if (dauer[bit]>maximum) maximum=dauer[bit];
					if (dauer[bit]<minimum) minimum=dauer[bit];
					summe=summe+dauer[bit];
					/* nachbrennzeit ermitteln */
					nachbrennen[bit]=faktor*dauer[bit];
					/* minimale brenndauer einhalten */
					if ((nachbrennen[bit]+dauer[bit])<min_impulse) {
						nachbrennen[bit]=min_impulse-dauer[bit];
						}
					dauer[bit]=0;
					} /* if bit gekippt */
				} /* for alle ergebnisbits */
			} while ((brennbyte!=255) && (!fehler));

		/* dieses byte ist jetzt gebrannt oder kaputt... */
		{
		float value;
		int anz=anzahl;
		int mini=minimum;if(minimum>maximum) mini=0;
		if (anz<1) anz=1;
		value=(1.0*summe)/(1.0*anz);
		if ((!(i%16))||(i==ende)) {printf("\r$%06lx (%06ld):  min=%2ld  max=%2ld avg=%g",
			i,i,mini,maximum,value);
			fflush(stdout);
			}
		}

		} /* for (alle zu brennenden adressen) */
	
	/* fertig */
	printf("\n");

	/* Programmierspannung auf 5V einstellen */
	SelectVoltage(5.0);	/* 5V ist "normale" spannung */
	cd4094Send(&u5);	/* U5 schreiben, setzt auch OE von U3 */
	usleep(1000000);	/* 1 sekunde warten */

	/* fertig */
	Done2764();
	printf("\nBitte Eprom 2764/27128 entnehmen und ENTER druecken");gets(s);
	fclose(f);
}



/*----------------------------------------------------------*/
/* 27256/27512 Routinen (partiell getestet)                 */
/*----------------------------------------------------------*/


/* vorbereitung zum einsezten des eproms */
void Init27256(){

	Reset4040();

	/* ausgaberegister egal */
	cd4094Write(&u3,0);	
	/* Steuerregister 4:  OE:=1, CE:=1 */
	cd4094Write(&u4,1+2);	

	/* Steuerregister 5:  Pin 1 ist statt Vpp A15, LED aus */
	/*SelectVoltage(5.0);*/
	cd4094Write(&u5,0);	
	Addr27256(0);
	LED(FALSE);
}



void Done27256(){

	/* ausgaberegister egal */
	cd4094Write(&u3,0);	

	/* Steuerregister 4:  OE:=1, CE:=1 */
	cd4094Write(&u4,1+2);	

	/* Steuerregister 5:  A15 statt Vpp an Pin 1, LED aus */
	cd4094Write(&u5,0);	
	Addr27256(0);
	LED(FALSE);
}



void Addr27256(long address){
long i;
int command;
volatile long high,low10;
volatile long act10,acthigh;

	/* -1 als aktuelle adresse besagt, dass sie auf jeden */
	/* fall neu gebildet werden muss! */
	if (act==-1) Reset4040();

/*	if (address==act) return;*/

	command=0;WriteCommand(command);

	/* adressbits trennen */	
	low10=address & (1+2+4+8+16+32+64+128+256+512);
	act10=act & (1+2+4+8+16+32+64+128+256+512);
	if (act10>low10) Reset4040();
	/* act10 neu berechnen */
	act10=act & (1+2+4+8+16+32+64+128+256+512);

	/* low part of address */
	for (i=act10;i<low10;i++) {
		/* clock strobe auf CLOCK fuer U1 (4040) erzeugen */
		WriteCommand(BIT_CLOCK1);
		WriteCommand(0);
		}

	high=address>>10;	
	acthigh=act>>10;	
	/* high part of address in U4 (4094) */

	u4.out[3]=(high & 1); 		/* bit 10 */
	u4.out[4]=(high & 2); 		/* bit 11 */
	u4.out[5]=(high & 4); 		/* bit 12 */
	u4.out[6]=!(high & 8); 		/* bit 13 ab 27128 (INVERS!) */
	u4.out[7]=(high & 16); 		/* bit 14 ab 27256 */
	if (Optionen.type==27512) {
		u5.out[0]=(high & 32); 	/* bit 15 ab 27512; u4[2] muss 0 sein! */
		}
	else {
		u5.out[0]=1; 		/* "Vpp" bei 27256 (hier: +5V) */
		}

	cd4094Send(&u4);
	cd4094Send(&u5);
	act=address;
}


 		

void Read27256(long start, long ende, char *filename){
char s[80];
long i;
int value;
FILE *f;

	assert(filename!=NULL);

	printf("Auslesen eines 27256/27512.\n");
	f=fopen(filename,"w");if (!f) return;

	Init27256();
	printf("Bitte Eprom 27256/27512 einsetzen und ENTER druecken");gets(s);

	/* LED einschalten */
	LED(TRUE);

	/* alle adressen auslesen */
	for (i=start;i<=ende;i++){
		Addr27256(i);
		value=ReadByteEprom();	/* von 4021 */
		putc(value,f);
		if (!(i%256)) {printf("\r$%06lx (%06ld): %02x",i,i,value);fflush(stdout);}
		}
	
	/* fertig */
	Done27256();
	printf("\nBitte Eprom 27256/27512 entnehmen und ENTER druecken");gets(s);
	fclose(f);
}

 

int Check27256(long start, long ende){
char s[80];
long i;
int value;
int result;
FILE *f;

	printf("Test, ob 27256/27512 leer ist.\n");
	result=TRUE;

	Init27256();
	printf("Bitte Eprom 27256/27512 einsetzen und ENTER druecken");gets(s);

	/* LED einschalten */
	LED(TRUE);

	/* alle adressen testen */
	for (i=start;i<=ende;i++){
		Addr27256(i);
		value=ReadByteEprom();	/* von 4021 */
		if (!(i%256)) {printf("\r$%06lx (%06ld): %02x",i,i,value);fflush(stdout);}
		if (value!=255) {
			result=FALSE;
			printf ("%08d: %03d\n",i,value);
			}
		}
	
	/* fertig */
	Done27256();
	printf("\nBitte Eprom 27256/27512 entnehmen und ENTER druecken");gets(s);

	if (result) {
		printf("Eprom 27256/27512 ist leer.\n");
		}
	else {
		printf("Eprom 27256/27512 ist NICHT leer!\n");
		}
	return(result);
}



/* vergleich file-eprom */
int Compare27256(long start, long ende, long offset, char *filename){
char s[80];
long i;
int value_r,value_w;
FILE *f;
int result;

	assert(filename!=NULL);

	printf("Vergleichen eines 27256/27512.\n");
	f=fopen(filename,"r");if (!f) return(FALSE);
	fseek(f,offset,SEEK_SET);
	result=TRUE;

	Init27256();
	printf("Bitte Eprom 27256/27512 einsetzen und ENTER druecken");gets(s);

	/* LED einschalten */
	LED(TRUE);

	/* alle adressen vergleichen */
	for (i=start;i<=ende;i++){
		value_w=fgetc(f);		/* byte lesen */
		Addr27256(i);
		value_r=ReadByteEprom();
		if (!(i%256)) {printf("\r$%06lx (%06ld): %02x",i,i,value_r);fflush(stdout);}
		if (value_w!=value_r) {
			printf ("difference at %08d [%06x]: read %03d [%02x] instead of %03d [%02x]\n",i,i,value_r,value_r,value_w,value_w);
			result=FALSE;
			}
		}
	
	/* fertig */
	Done27256();
	printf("\nBitte Eprom 27256/27512 entnehmen und ENTER druecken");gets(s);
	fclose(f);
	return(result);
}


/* standard-algorithmus, aber $FF werden uebersprungen */
void Write27256(long start, long ende, long offset, char *filename, int num_impulse, long impuls_breite){
char s[80];
long i,j;
int value_r,value_w;
FILE *f;
tiostruct parms;

	assert(filename!=NULL);
	printf("Brennen eines 27256/27512.\n");

	f=fopen(filename,"r");if (!f) return;
	fseek(f,offset,SEEK_SET);

	Init27256();
	printf("Bitte Eprom 27256/27512 einsetzen und ENTER druecken");gets(s);

	/* LED einschalten */
	LED(TRUE);

/* programmiermodus einschalten */		

	/* Eprom Pin 1 einstellen: Vpp oder A15 */
	if (Optionen.type==27256) {
		u5.out[4]=1;		/* '256: Pin1 ist Vpp */
		u5.out[0]=1;		/* "A15" (egal, wird uebersteuert von Vpp) */
		u4.out[2]=0;		/*  /OE bisherigen /OE erhalten */
		}
	else if (Optionen.type==27512) {
		u5.out[4]=0;		/* '512: Pin1 ist A15 */
		u5.out[0]=0;		/* A15 */
		u4.out[2]=1;		/* 512: /OE muss Vpp erhalten */
		}
	else {
		printf("falscher typ! bitte eprom entnehmen! \n");
		LED(FALSE);
		return;
		}

	/* spannung waehlen */
	SelectVoltage(Optionen.voltage);	
	cd4094Send(&u5);	/* U5 schreiben */
	usleep(1000000);	/* 1 sekunde warten */
	cd4094Send(&u4);	/* U4 schreiben */

	/* port setzen (einmal reicht) */
	parms.port=LPTBASE;
	parms.delay=impuls_breite-us_error;

	printf("\nprogramming:\n");

	/* alle adressen schreiben */
	for (i=start;i<=ende;i++){
		if (!(i%16)) {printf("\r$%06lx (%ld)",i,i);fflush(stdout);}
		value_w=fgetc(f);		/* byte von file lesen */
		Addr27256(i);			/* adresse an 27256 einstellen */

		/* wert -> 4094 */
		cd4094Write(&u3,value_w);
		U3OE=U3OE_ON;		/* OE von U5 (4094) enable bei naechstem Kommando! */
		
		/* programmieren: /OE bleibt high, /CE geht auf low! */
		cd4094Send(&u4);		/* /CE = 1 schreiben, setzt auch OE von U3 */

		parms.wert1=cd4094Read(&u4) & 254;	/* /CE -> 0 */
		parms.wert2=cd4094Read(&u4) | 1;	/* /CE -> 1 */

		/* innere "brennschleife" */
		if (value_w!=255) for (j=0;j<num_impulse;j++) {
			/* schreibt werte[0], wartet 1000us, schreibt werte[1] */
			if (ioctl(tio_id,TIO_PROG,&parms));
			} /* for j */

		U3OE=U3OE_OFF;		/* OE von U5 (4094) disable bei naechstem Kommando! */
		} /* for (alle zu brennenden adressen) */

/* fertig */
	printf("\n");

/* programmiermodus abschalten */

	/* Eprom Pin 1 abstellen: Vpp oder A15 */
	if (Optionen.type==27256) {
		u5.out[4]=0;		/*  Pin1 ist wieder A15 */
		u5.out[0]=0;		/* '256: Pin1 "A15" abstellen */
		}
	else if (Optionen.type==27512) {
		u5.out[4]=0;		/* '512: Pin1 ist A15 */
		u5.out[0]=0;		/* A15 */
		u4.out[2]=1;		/* 512: /OE muss wieder /OE erhalten statt Vpp */
		}
	else {
		printf("falscher typ! \n");
		}


	SelectVoltage(5.0);	/* 5V ist "normale" spannung */
	cd4094Send(&u5);	/* U5 schreiben, setzt auch OE von U3 */
	cd4094Send(&u4);	/* U4 schreiben */
	usleep(1000000);	/* 1 sekunde warten */

	/* fertig */
	Done27256();
	printf("\nBitte Eprom 27256/27512 entnehmen und ENTER druecken");gets(s);
	fclose(f);
}



void FastWrite27256(long start,long ende,long offset,char *filename,
	int min_impulse,int max_impulse, int faktor, long impuls_breite){
char s[80];
long i,j;
int value_r,value_w;
FILE *f;
bool	bits_in[8],bits_out[8];
int	dauer[8];
int	nachbrennen[8];
int	minimum,maximum,anzahl,bit,r_alt,r_neu;
unsigned int brennbyte;
bool	fehler;
long summe;
tiostruct parms;

	assert(filename!=NULL);
	printf("Brennen eines 27256/27512 mit Quick 1.\n");

	f=fopen(filename,"r");if (!f) return;
	fseek(f,offset,SEEK_SET);

	for (i=0;i<8;i++) {
		bits_in[i]=FALSE;
		bits_out[i]=FALSE;
		dauer[i]=0;
		nachbrennen[i]=0;
		}
	maximum=0;summe=0;minimum=99;anzahl=0;

	Init27256();
	printf("Bitte Eprom 27256/27512 einsetzen und ENTER druecken");gets(s);

	/* LED einschalten */
	LED(TRUE);

	/* Programmierspannung xxV  waehlen */
	SelectVoltage(Optionen.voltage);	
	cd4094Send(&u5);	/* U5 schreiben */
	usleep(1000000);	/* 1 sekunde warten */

	/* port setzen (einmal reicht) */
	parms.port=LPTBASE;
	parms.delay=impuls_breite-us_error;

	printf("\nprogramming:\n");
	fehler=FALSE;

	/* alle adressen schreiben */
	for (i=start;(i<=ende)&&!fehler;i++){
		value_w=fgetc(f);		/* byte von file lesen */
		Addr27256(i);			/* adresse an 27256 einstellen */

		/* 0. testen, ob das byte ueberhaupt programmierbar ist */
		value_r=ReadByteEprom(); 

		if (value_r!=255) {
			fehler=TRUE;
			printf("\nFehler: Byte %d ist nicht $FF sondern $%02x!\n",i,value_r);
			}

		/* 1. feststellen, welche bits von value_w auf 0 zu bringen sind */
		for (bit=7;bit>=0;bit--) {
			if (value_w&(1<<bit)) {
				dauer[bit]=0;
				}
			else {
				dauer[bit]=1;
				anzahl++;		/* 0-Bits zaehlen */
				}
			nachbrennen[bit]=0;
			}

		/* 2. das aktuelle byte an aktueller adresse brennen */
		do {
			/* aktuelles byte aus dem eprom lesen */
			r_alt=ReadByteEprom();
if (debugflag) fprintf(stderr,"Gelesen1[$%04x]: $%02x\n",i,r_alt);
			/* zu brennendes byte erstellen */
			brennbyte=0;
			for (bit=7;bit>=0;bit--) {
				brennbyte=brennbyte<<1;	
				if (nachbrennen[bit]) nachbrennen[bit]--;
				if (dauer[bit]) dauer[bit]++;
				if (dauer[bit] || nachbrennen[bit]) brennbyte++;
				if ((dauer[bit])>max_impulse) {
					printf("\nFehler: $%02x statt $%02x (bit %d)\n",
						r_alt,value_w,bit);
					fehler=TRUE;
					}
				}
			brennbyte=(~brennbyte)&255;	/* gesetzte bits auf 0 brennen */

			/* das byte brennen, $FF immer ueberspringen */
			if (brennbyte!=255) {
				/* wert -> 4094 */
				cd4094Write(&u3,brennbyte);
				U3OE=U3OE_ON;	/* OE von U5 (4094) enable bei next cmd */

				/* Eprom Pin 1 einstellen: Vpp oder A15 */
				if (Optionen.type==27256) {
					u5.out[4]=1;		/* '256: Pin1 ist Vpp */
					u4.out[2]=0;		/* OE muss bisherigen OE erhalten*/
					}
				else if (Optionen.type==27512) {
					u5.out[4]=0;		/* '512: Pin1 ist A15 */
					u4.out[2]=1;		/* 512: /OE muss Vpp erhalten */
					}
				cd4094Send(&u4);	/* U4 schreiben */
				cd4094Send(&u5);	/* U5 schreiben */

				parms.wert1=cd4094Read(&u4) & 254;	/* /CE -> 0 */
				parms.wert2=cd4094Read(&u4) | 1;	/* /CE -> 1 */

#ifdef thh
				u5.out[0]=1;		/* Pin1 von Eprom auf Vpp setzen */
				cd4094Send(&u5);	/* U5 schreiben */
#endif

				/* schreibt wert1, wartet n us, schreibt wert2 */
				if (ioctl(tio_id,TIO_PROG,&parms));

				U3OE=U3OE_OFF;	/* OE von U5 (4094) disable bei next cmd */

				/* Eprom Pin 1 abstellen: Vpp oder A15 */
				if (Optionen.type==27256) {
					u5.out[4]=0;		/*  Pin1 ist "A15" */
					}
				else if (Optionen.type==27512) {
					u5.out[4]=0;		/* '512: Pin1 ist A15 */
					u4.out[2]=0;		/* 512: /OE statt Vpp an /OE */
					}
				cd4094Send(&u4);	/* U4 schreiben */
				cd4094Send(&u5);	/* U5 schreiben */

if (debugflag) fprintf(stderr,"Gebrannt[$%04x]: $%02x\n",i,brennbyte);

				} /* if brennbyte */

			/* ergebnis pruefen */
			r_neu=ReadByteEprom();
if (debugflag) fprintf(stderr,"Gelesen2[$%04x]: $%02x\n",i,r_neu);
			for (bit=7;bit>=0;bit--) {
				int r_exor=(r_alt ^ r_neu);
				if ((!nachbrennen[bit]) && (r_exor&(1<<bit))) {
					/* ja, dieses bit ist gerade jetzt gekippt! */
					dauer[bit]--;	/* dauer ist immer 1 zu gross! */
					if (dauer[bit]>maximum) maximum=dauer[bit];
					if (dauer[bit]<minimum) minimum=dauer[bit];
					summe=summe+dauer[bit];
					/* nachbrennzeit ermitteln */
					nachbrennen[bit]=faktor*dauer[bit];
					/* minimale brenndauer einhalten */
					if ((nachbrennen[bit]+dauer[bit])<min_impulse) {
						nachbrennen[bit]=min_impulse-dauer[bit];
						}
					dauer[bit]=0;
					} /* if bit gekippt */
				} /* for alle ergebnisbits */
			} while ((brennbyte!=255) && (!fehler));

		/* dieses byte ist jetzt gebrannt oder kaputt... */
		{
		float value;
		int anz=anzahl;
		int mini=minimum;if(minimum>maximum) mini=0;
		if (anz<1) anz=1;
		value=(1.0*summe)/(1.0*anz);
		if ((!(i%16))||(i==ende)) {printf("\r$%06lx (%06ld):  min=%2ld  max=%2ld avg=%g",
			i,i,mini,maximum,value);
			fflush(stdout);
			}
		}

		} /* for (alle zu brennenden adressen) */
	
	/* fertig */
	printf("\n");

	/* Programmierspannung auf 5V einstellen */
	SelectVoltage(5.0);	/* 5V ist "normale" spannung */
	cd4094Send(&u5);	/* U5 schreiben, setzt auch OE von U3 */
	usleep(1000000);	/* 1 sekunde warten */

	/* fertig */
	Done27256();
	printf("\nBitte Eprom 27256/27512 entnehmen und ENTER druecken");gets(s);
	fclose(f);
}



/*----------------------------------------------------------*/
/* 2716/2732 Routinen (ungetestet)                        */
/*----------------------------------------------------------*/

/* vorbereitung zum einsezten des eproms */
void Init2716(){

	Reset4040();

	/* ausgaberegister egal */
	cd4094Write(&u3,0);	

	/* Steuerregister 4:  OE:=1, CE:=1, PGM==A14:=1 */
	/* bit 6 schaltet vcc an pin 24 des `16 (d.h. pin 26 von textool */
	/* achtung: bit ist invers! 64=log 0 */
	cd4094Write(&u4,1+2);	

	/* Steuerregister 5:  alles auf 0 */
	/*SelectVoltage(5.0);*/
	cd4094Write(&u5,0);	
	Addr2716(0);
	LED(FALSE);
}



void Done2716(){

	/* ausgaberegister egal */
	cd4094Write(&u3,0);	

	/* Steuerregister 4:  OE:=1, CE:=1, PGM==A14:=1 */
	/* bit 6 schaltet vcc an pin 24 des `16 (d.h. pin 26 von textool) */
	/* achtung: bit ist invers! 64=log 0 */
	cd4094Write(&u4,1+2);	

	/* Steuerregister 5:  Vpp an Pin 1 auf 5V, LED aus */
	cd4094Write(&u5,16);	
	Addr2716(0);
	LED(FALSE);
}



void Addr2716(long address){
long i;
int command;
volatile long high,low10;
volatile long act10,acthigh;

	/* -1 als aktuelle adresse besagt, dass die auf jeden */
	/* fall neu gebildet werden muss! */
	if (act==-1) Reset4040();

/*	if (address==act) return;*/

	command=0;WriteCommand(command);

	/* adressbits trennen */	
	low10=address & (1+2+4+8+16+32+64+128+256+512);
	act10=act & (1+2+4+8+16+32+64+128+256+512);
	if (act10>low10) Reset4040();
	/* act10 neu berechnen */
	act10=act & (1+2+4+8+16+32+64+128+256+512);

	/* low part of address */
	for (i=act10;i<low10;i++) {
		/* clock strobe auf CLOCK fuer U1 (4040) erzeugen */
		WriteCommand(BIT_CLOCK1);
		WriteCommand(0);
		}

	high=address>>10;	
	acthigh=act>>10;	

	/* high part of address in U4 (4094) */
	u4.out[3]=(high & 1); 		/* bit 10 */
	/* Achtung: laut ct 3/88 S. 244 sollten hier +5V anliegen, */
	/* nicht nur CMOS-High - 0.7V !*/
	u4.out[4]=(high & 2); 		/* bit 11 ab 2732 */
	/* wenn es ein 2716 ist, handelt es sich exklusiv um Vpp */
	if (Optionen.type==2716) u4.out[4]=1;	

	cd4094Send(&u4);
	cd4094Send(&u5);
	act=address;
}



void Read2716(long start, long ende, char *filename){
char s[80];
long i;
int value;
FILE *f;

	assert(filename!=NULL);

	printf("Auslesen eines 2716/2732.\n");
	f=fopen(filename,"w");if (!f) return;

	Init2716();
	printf("Bitte Eprom 2716/2732 einsetzen und ENTER druecken");gets(s);

	/* LED einschalten */
	LED(TRUE);

	/* alle adressen auslesen */
	for (i=start;i<=ende;i++){
		Addr2716(i);
		value=ReadByteEprom();	/* von 4021 */
		putc(value,f);
		if (!(i%256)) {printf("\r$%06lx (%06ld): %02x",i,i,value);fflush(stdout);}
		}
	
	/* fertig */
	Done2716();
	printf("\nBitte Eprom 2716/2732 entnehmen und ENTER druecken");gets(s);
	fclose(f);
}

 

int Check2716(long start, long ende){
char s[80];
long i;
int value;
int result;
FILE *f;

	printf("Test, ob 2716/2732 leer ist.\n");
	result=TRUE;

	Init2716();
	printf("Bitte Eprom 2716/2732 einsetzen und ENTER druecken");gets(s);

	/* LED einschalten */
	LED(TRUE);

	/* alle adressen testen */
	for (i=start;i<=ende;i++){
		Addr2716(i);
		value=ReadByteEprom();	/* von 4021 */
		if (!(i%256)) {printf("\r$%06lx (%06ld): %02x",i,i,value);fflush(stdout);}
		if (value!=255) {
			result=FALSE;
			printf ("%08d: %03d\n",i,value);
			}
		}
	
	/* fertig */
	Done2716();
	printf("\nBitte Eprom 2716/2732 entnehmen und ENTER druecken");gets(s);

	if (result) {
		printf("Eprom 2716/2732 ist leer.\n");
		}
	else {
		printf("Eprom 2716/2732 ist NICHT leer!\n");
		}
	return(result);
}



/* vergleich file-eprom */
int Compare2716(long start, long ende, long offset, char *filename){
char s[80];
long i;
int value_r,value_w;
FILE *f;
int result;

	assert(filename!=NULL);

	printf("Vergleichen eines 2716/2732.\n");
	f=fopen(filename,"r");if (!f) return(FALSE);
	fseek(f,offset,SEEK_SET);
	result=TRUE;

	Init2716();
	printf("Bitte Eprom 2716/2732 einsetzen und ENTER druecken");gets(s);

	/* LED einschalten */
	LED(TRUE);

	/* alle adressen vergleichen */
	for (i=start;i<=ende;i++){
		value_w=fgetc(f);		/* byte lesen */
		Addr2716(i);
		value_r=ReadByteEprom();
		if (!(i%256)) {printf("\r$%06lx (%06ld): %02x",i,i,value_r);fflush(stdout);}
		if (value_w!=value_r) {
			printf ("difference at %08d [%06x]: read %03d [%02x] instead of %03d [%02x]\n",i,i,value_r,value_r,value_w,value_w);
			result=FALSE;
			}
		}
	
	/* fertig */
	Done2716();
	printf("\nBitte Eprom 2716/2732 entnehmen und ENTER druecken");gets(s);
	fclose(f);
	return(result);
}



/*----------------------------------------------------------*/
/* adjust                                                   */
/*----------------------------------------------------------*/


void Adjust(){
char s[80];

	printf("Justage/Test der Programmierspannungen.\n");
	printf("Waehrend des Tests sollte die LED leuchten.\n");
	printf(" --- KEIN EPROM EINSETZEN! ---\n");

	Init2764();

	/* LED einschalten */
	LED(TRUE);

	/* Programmierspannung xxV an Eprom Pin 1 einstellen */
	u5.out[0]=1;		/* Pin1 von Eprom setzen */

	/* test: 5V */
	SelectVoltage(5.0);	
	cd4094Send(&u5);	/* U5 schreiben */
	usleep(1000000);	/* 1 sekunde warten */
	printf("An Pin 1 des Textoolsockels muessen jetzt +5.0V anliegen <ENTER>");
	gets(s);

	/* test: 12.5V */
	SelectVoltage(12.5);	
	cd4094Send(&u5);	/* U5 schreiben */
	usleep(1000000);	/* 1 sekunde warten */
	printf("An Pin 1 des Textoolsockels muessen jetzt +12.5V anliegen <ENTER>");
	gets(s);

	/* test: 21V */
	SelectVoltage(21.0);	
	cd4094Send(&u5);	/* U5 schreiben */
	usleep(1000000);	/* 1 sekunde warten */
	printf("An Pin 1 des Textoolsockels muessen jetzt +21.0V anliegen <ENTER>");
	gets(s);

	/* test: 25V */
	SelectVoltage(25.0);	
	cd4094Send(&u5);	/* U5 schreiben */
	usleep(1000000);	/* 1 sekunde warten */
	printf("An Pin 1 des Textoolsockels muessen jetzt +25.0V anliegen <ENTER>");
	gets(s);

	/* tests beendet */

	/* Programmierspannung 5V/TTL an Eprom Pin 1 einstellen */
	SelectVoltage(5.0);	
	u5.out[0]=0;		/* Pin1 von Eprom setzen */
	cd4094Send(&u5);	/* U5 schreiben */
	usleep(1000000);	/* 1 sekunde warten */

	/* LED abschalten */
	LED(FALSE);

	/* fertig */
}



/*----------------------------------------------------------*/
/* debug                                                    */
/*----------------------------------------------------------*/

void debug(){
char eingabe[80];
int device,value;

	while(TRUE){
		printf("\nchip (3..5)? ");
		gets(eingabe);
		device=atoi(eingabe);
		if (device==0) return;
		printf("wert (0..255)? ");
		gets(eingabe);
		value=atoi(eingabe);
		if (device==3) cd4094Write(&u3,value);
		if (device==4) cd4094Write(&u4,value);
		if (device==5) cd4094Write(&u5,value);
		}
}


/*----------------------------------------------------------*/
/* menuesystem                                              */
/*----------------------------------------------------------*/

/* defaults */
void InitOptions(){
	Optionen.type=2764;
	Optionen.size=8192;
	strcpy(Optionen.type_name,"2764");
	Optionen.voltage=25.0;
	strcpy(Optionen.voltage_name,"25.0 V");
	Optionen.algorithm=1;
	strcpy(Optionen.algorithm_name,"Standard 50*1ms");
	Optionen.port=LPT2;
	strcpy(Optionen.port_name,"LPT2");
	/* user interface */
	Optionen.start=0;
	Optionen.ende=Optionen.size-1;
	Optionen.offset=0;
	strcpy(Optionen.filename,"image.rom");
	/* custom routinen */
	Optionen.c_delay=1000;	
	Optionen.c_min=1;	
	Optionen.c_max=25;	
	Optionen.c_faktor=3;
	Optionen.c_sdelay=1000;	
	Optionen.c_snum=50;	
}


void Header(){

	printf("\n\n\nprommer V1.0 fuer REX Datentechnik Quickbyte 2.0 (Linux/80x86)\n");
	printf("[%s] [Alg: %s] [Vpp: %s] [%s]\n\n",
		Optionen.type_name,Optionen.algorithm_name,
		Optionen.voltage_name,Optionen.port_name);
}


int GetNum(){
char s[80];
	printf("\n?. Wahl ");
	gets(s);
	return(atoi(s));
}



long GetLong(char *name, long deflt){
char s[80];

	assert(name!=NULL);

	printf("%s (default: %ld): ",name,deflt);
	gets(s);
	if (strlen(s)>0) {
		return(atol(s));
		}
	else {
		return(deflt);
		}
}



char *GetString(char *name, char *deflt){
static char s[1024];

	assert(name!=NULL);
	printf("%s (default: %s): ",name,deflt);
	gets(s);
	if (strlen(s)>0) {
		/* achtung: geht nur, wenn s static ist! */
		return(s);
		}
	else {
		return(deflt);
		}
}



void Copyright(){
char s[80];

	Header();

	printf("Gewaehrleistung\n");
	printf("===============\n");
	printf("Dieses Programm kommt OHNE JEGLICHE GEWAEHRLEISTUNG.\n");
	printf("Es wird keine Haftung uebernommen, weder fuer eine\n");
	printf("einwandfreie Funktion noch fuer auftretende Schaeden \n");
	printf("an Hard- oder Software.\n");

	printf("\n");

	printf("Copyright\n");
	printf("=========\n");
	printf("Das Copyright fuer dieses Tool liegt bei thh-computergraphics.\n");
	printf("Die Nutzung ist kostenlos. Fuer das Kopieren gelten\n");
	printf("die Bestimmungen der GNU Public License (GPL).\n");

	printf("\n");

	printf("Autor\n");
	printf("=====\n");
	printf("Thorsten Hempel\n");
	printf("Meisterstr. 23\n");
	printf("23919 Berkenthin\n");
	printf("thh@valis.pumuckl.cubenet.de\n");

	printf("\n");

	printf("Ok. ");gets(s);
}


void Epromtyp(){
int i=-1;

	while (i!=0) {
		Header();

		printf("Epromtyp\n");
		printf("========\n");
		printf("0. Zurueck\n");
		printf("1. 2716  (Programmierung noch nicht unterstuetzt)\n");
		printf("2. 2732  (Programmierung noch nicht unterstuetzt)\n");
		printf("3. 2764\n");
		printf("4. 27128 (noch nicht getestet)\n");
		printf("5. 27256 (noch nicht getestet)\n");
		printf("6. 27512\n");

		i=GetNum();
		if (i==1) {
			Optionen.type=2716;
			Optionen.size=2048;
			strcpy(Optionen.type_name,"2716");
			}
		else if (i==2) {
			Optionen.type=2732;
			Optionen.size=4096;
			strcpy(Optionen.type_name,"2732");
			}
		else if (i==3) {
			Optionen.type=2764;
			Optionen.size=8192;
			strcpy(Optionen.type_name,"2764");
			}
		else if (i==4) {
			Optionen.type=27128;
			Optionen.size=16384;
			strcpy(Optionen.type_name,"27128");
			}
		else if (i==5) {
			Optionen.type=27256;
			Optionen.size=32768;
			strcpy(Optionen.type_name,"27256");
			}
		else if (i==6) {
			Optionen.type=27512;
			Optionen.size=65536;
			strcpy(Optionen.type_name,"27512");
			}
		if (Optionen.ende>=Optionen.size) Optionen.ende=Optionen.size-1;
		if (Optionen.start>Optionen.ende) Optionen.start=0;
		if (Optionen.offset+Optionen.start>Optionen.ende) Optionen.offset=0;
		}
}


void Algorithmus(){
int i=-1;

	while (i!=0) {
		Header();

		printf("Brennalgorithmus\n");
		printf("================\n");
		printf("0. Zurueck\n");
		printf("1. Standard   : (50)      1 ms Impulse \n");
		printf("2. Quick 1    : (n+3n)    1 ms Impulse, 1<=n<=25 \n");
		printf("3. Quick 2    : (n+3n)  200 us Impulse, 1<=n<=25 \n");
		printf("\n");
		printf("4. Custom ('Standard'): (%2ld)%7ld us Impulse \n",
			Optionen.c_snum,Optionen.c_sdelay);
		printf("5. Custom ('Quick')   : (n+%ldn)%5ld us Impulse, %ld<=n<%ld\n",
			Optionen.c_faktor,Optionen.c_delay,Optionen.c_min,Optionen.c_max);

		i=GetNum();
		if (i==1) {
			Optionen.algorithm=1;
			strcpy(Optionen.algorithm_name,"Standard 50*1ms");
			}
		else if (i==2) {
			Optionen.algorithm=2;
			strcpy(Optionen.algorithm_name,"Quick 1");
			}
		else if (i==3) {
			Optionen.algorithm=3;
			strcpy(Optionen.algorithm_name,"Quick 2");
			}
		else if (i==4) {
			Optionen.algorithm=4;
			Optionen.c_sdelay=GetLong("Impulsbreite (in us): ",Optionen.c_sdelay);
			Optionen.c_snum=GetLong("Anzahl der Programmierimpulse: ",Optionen.c_snum);
			sprintf(Optionen.algorithm_name,"Standard %ld * %ld us",
				Optionen.c_snum,Optionen.c_sdelay);
			}
		else if (i==5) {
			Optionen.algorithm=5;
			Optionen.c_delay=GetLong("Impulsbreite (in us): ",
				Optionen.c_delay);
			Optionen.c_min=GetLong("Minimale Anzahl an Programmierimpulsen: ",
				Optionen.c_min);			
			Optionen.c_max=GetLong("Maximale Anzahl an Programmierimpulsen: ",
				Optionen.c_max);
			Optionen.c_faktor=GetLong("Faktor zur Nachprogrammierung: ",
				Optionen.c_faktor);
			sprintf(Optionen.algorithm_name,
				"Quick n+n*%ld, %ld us, %ld<=n<%ld",
				Optionen.c_faktor,Optionen.c_delay,Optionen.c_min,Optionen.c_max);
			}
		} /* while */
}



void Spannung(){
int i=-1;

	while (i!=0) {
		Header();

		printf("Programmierspannung\n");
		printf("===================\n");
		printf("0. Zurueck\n");
		printf("1.  5.0 V\n");
		printf("2. 12.5 V\n");
		printf("3. 21.0 V\n");
		printf("4. 25.0 V\n");

		i=GetNum();
		if (i==1) {
			Optionen.voltage=5.0;
			strcpy(Optionen.voltage_name,"5.0 V");
			}
		else if (i==2) {
			Optionen.voltage=12.5;
			strcpy(Optionen.voltage_name,"12.5 V");
			}
		else if (i==3) {
			Optionen.voltage=21.0;
			strcpy(Optionen.voltage_name,"21.0 V");
			}
		else if (i==4) {
			Optionen.voltage=25.0;
			strcpy(Optionen.voltage_name,"25.0 V");
			}
		}
}



void Port(){
int i=-1;
long oldport;
int err;

	while (i!=0) {
		oldport=Optionen.port;
		Header();

		printf("IO Port Basisadresse des Eprommers\n");
		printf("==================================\n");
		printf("0. Zurueck\n");
		printf("1. $378 ('LPT1')\n");
		printf("2. $278 ('LPT2')\n");
		printf("3. $3bc ('LPT3')\n");
		printf("4. $2bc ('LPT4')\n");

		i=GetNum();
		if (i==1) {
			Optionen.port=LPT1;
			strcpy(Optionen.port_name,"LPT1");
			}
		else if (i==2) {
			Optionen.port=LPT2;
			strcpy(Optionen.port_name,"LPT2");
			}
		else if (i==3) {
			Optionen.port=LPT3;
			strcpy(Optionen.port_name,"LPT3");
			}
		else if (i==4) {
			Optionen.port=LPT4;
			strcpy(Optionen.port_name,"LPT4");
			}
		}

	/* alte IOPERM  freigeben */
	ioperm(oldport,3,0);

	/* neue IOPERM setzen */
	if (ioperm(LPTBASE,3,1)) {
		err=errno;
		fprintf(stderr,"ioperm($%lx,3,1) failed (%d): %s\n",LPTBASE,err,strerror(err));
		if (err==1) fprintf(stderr,"(this program requires root privilges)\n");
		exit(1);
		}
}



void Einstellungen(){
int i=-1;

	while (i!=0) {
		Header();

		printf("Einstellungen\n");
		printf("=============\n");
		printf("0. Zurueck\n");
		printf("1. Epromtyp\n");
		printf("2. Brennalgorithmus\n");
		printf("3. Programmierspannung\n");
		printf("4. IO Port\n");
		printf("5. Spannungen testen\n");

		i=GetNum();
		if      (i==1) Epromtyp();
		else if (i==2) Algorithmus();
		else if (i==3) Spannung();
		else if (i==4) Port();
		else if (i==5) Adjust();
		}
}


/* withfilename fragt auch nach einem dateinamen */
/* withoffset fragt auch nach einem dateioffset (nur mit dateinamen sinnvoll) */
int GetRange(int withfilename,int withoffset){
	/* eingabe */
	Optionen.start=GetLong("Startadresse: ",Optionen.start);
	Optionen.ende=GetLong("Endadresse (-1 fuer Maximum): ",Optionen.ende);
	if (withfilename) {
		strcpy(Optionen.filename,GetString("Filename: ",Optionen.filename));
		if (withoffset) {
			Optionen.offset=GetLong("Offset in Datei: ",Optionen.start);
			}
		}	

	/* eingaben ueberpruefen */
	if (Optionen.start<0) return(FALSE);
	/* kurzform: -1 ist die letzte moegliche adresse im eingestellten eprom! */
	if (Optionen.ende<0) Optionen.ende=Optionen.size+Optionen.ende;
	if (Optionen.ende<Optionen.start) return(FALSE);
	if (Optionen.ende>Optionen.size) return(FALSE);
	if (withfilename) {
		if (strlen(Optionen.filename)<1) return(FALSE);
		if (withoffset) {
			if (Optionen.offset<0) return(FALSE);
			}
		}
	return(TRUE);
}



void Brennen(){
char *name;

	if (Optionen.type<2764) return;

	printf("\n\n");
	Header();
	printf("Programmieren/Brennen eines Eproms (%s)\n",Optionen.type_name);
	if (!GetRange(TRUE,TRUE)) return;

	if (Optionen.type>27128) {
		if (Optionen.algorithm==1) {
			Write27256(Optionen.start,Optionen.ende,Optionen.offset,Optionen.filename,
				50,1000);
			}
		else if (Optionen.algorithm==2) {
			FastWrite27256(Optionen.start,Optionen.ende,Optionen.offset,
				Optionen.filename,1,25,3,1000);
			}
		else if (Optionen.algorithm==3) {
			FastWrite27256(Optionen.start,Optionen.ende,Optionen.offset,
				Optionen.filename,1,25,3,200);
			}
		else if (Optionen.algorithm==4) {
			Write27256(Optionen.start,Optionen.ende,Optionen.offset,Optionen.filename,
				Optionen.c_snum,Optionen.c_sdelay);
			}
		else if (Optionen.algorithm==5) {
			FastWrite27256(Optionen.start,Optionen.ende,Optionen.offset,
				Optionen.filename,Optionen.c_min,Optionen.c_max,
				Optionen.c_faktor,Optionen.c_delay);
			}
		else {
			printf("Eingestellter Algorithmus nicht implementiert!\n");
			}
		return;
		} /* 256/512 */

	if (Optionen.algorithm==1) {
		Write2764(Optionen.start,Optionen.ende,Optionen.offset,Optionen.filename,
			50,1000);
		}
	else if (Optionen.algorithm==2) {
		FastWrite2764(Optionen.start,Optionen.ende,Optionen.offset,Optionen.filename,
			1,25,3,1000);
		}
	else if (Optionen.algorithm==3) {
		FastWrite2764(Optionen.start,Optionen.ende,Optionen.offset,Optionen.filename,
			1,25,3,200);
		}
	else if (Optionen.algorithm==4) {
		Write2764(Optionen.start,Optionen.ende,Optionen.offset,Optionen.filename,
			Optionen.c_snum,Optionen.c_sdelay);
		}
	else if (Optionen.algorithm==5) {
		FastWrite2764(Optionen.start,Optionen.ende,Optionen.offset,Optionen.filename,
			Optionen.c_min,Optionen.c_max,Optionen.c_faktor,Optionen.c_delay);
		}
	else {
		printf("Eingestellter Algorithmus nicht implementiert!\n");
		}
}



void Auslesen(){
char *name;
	printf("\n\n");
	Header();
	printf("Auslesen des Eproms (%s)\n",Optionen.type_name);
	if (!GetRange(TRUE,FALSE)) return;

	if ((Optionen.type==2764) || (Optionen.type==27128)) { 
		Read2764(Optionen.start,Optionen.ende,Optionen.filename);
		}
	else if ((Optionen.type==27256) || (Optionen.type==27512)) { 
		Read27256(Optionen.start,Optionen.ende,Optionen.filename);
		}
	else if ((Optionen.type==2716) || (Optionen.type==2732)) { 
		Read2716(Optionen.start,Optionen.ende,Optionen.filename);
		}
}
  

void Vergleichen(){
char *name;
char s[80];

	printf("\n\n");
	Header();
	printf("Vergleichen eines Eproms (%s) mit einer Datei\n",Optionen.type_name);
	if (!GetRange(TRUE,TRUE)) return;

	if ((Optionen.type==2764) || (Optionen.type==27128)) { 
		Compare2764(Optionen.start,Optionen.ende,Optionen.offset,Optionen.filename);
		}
	else if ((Optionen.type==27256) || (Optionen.type==27512)) { 
		Compare27256(Optionen.start,Optionen.ende,Optionen.offset,Optionen.filename);
		}
	else if ((Optionen.type==2716) || (Optionen.type==2732)) { 
		Compare2716(Optionen.start,Optionen.ende,Optionen.offset,Optionen.filename);
		}
}

 

void Testen(){
char *name;
char s[80];
int result;

	printf("\n\n");
	Header();
	printf("Testen des Eproms (%s) auf Programmierbarkeit ($FF)\n",Optionen.type_name);
	if (!GetRange(FALSE,FALSE)) return;

	if ((Optionen.type==2764) || (Optionen.type==27128)) { 
		result=Check2764(Optionen.start,Optionen.ende);
		}
	else if ((Optionen.type==27256) || (Optionen.type==27512)) { 
		result=Check27256(Optionen.start,Optionen.ende);
		}
	else if ((Optionen.type==2716) || (Optionen.type==2732)) { 
		result=Check2716(Optionen.start,Optionen.ende);
		}
	else {
		result=FALSE;
		}
	if (!result) {
		printf("Check(%ld,%ld) lieferte FALSE.\n",Optionen.start,Optionen.ende);
		}
	else {
		printf("Check(%ld,%ld) lieferte OK.\n",Optionen.start,Optionen.ende);
		}
	gets(s);
}



void split(){
char name81[1024],name82[1924],name16[1024];
FILE *f81,*f82,*f16;
int c81,c82;

	strcpy(name16,GetString("      16-Bit Datei: ","image16.rom"));
	strcpy(name81,GetString("Erste  8-Bit Datei: ","image8-1.rom"));
	strcpy(name82,GetString("Zweite 8-Bit Datei: ","image8-2.rom"));

	f81=fopen(name81,"w");if (!f81) return;
	f82=fopen(name82,"w");if (!f82) {fclose(f81);return;}
	f16=fopen(name16,"r");if (!f16) {fclose(f81);fclose(f82);return;}

	while (((c81=fgetc(f16))!=EOF) && ((c82=fgetc(f16))!=EOF)) { 
		fputc(c81,f81);
		fputc(c82,f82);
		}
	fclose(f81);
	fclose(f82);
	fclose(f16);
}


void join(){
char name81[1024],name82[1924],name16[1024];
FILE *f81,*f82,*f16;
int c81,c82;

	strcpy(name81,GetString("Erste  8-Bit Datei: ","image8-1.rom"));
	strcpy(name82,GetString("Zweite 8-Bit Datei: ","image8-2.rom"));
	strcpy(name16,GetString("      16-Bit Datei: ","image16.rom"));

	f81=fopen(name81,"r");if (!f81) return;
	f82=fopen(name82,"r");if (!f82) {fclose(f81);return;}
	f16=fopen(name16,"w");if (!f16) {fclose(f81);fclose(f82);return;}

	while (((c81=fgetc(f81))!=EOF) && ((c82=fgetc(f82))!=EOF)) { 
		fputc(c81,f16);
		fputc(c82,f16);
		}
	fclose(f81);
	fclose(f82);
	fclose(f16);
}



void Hauptmenue(){
int i=-1;

	InitOptions();

	while (i!=0) {
		Header();

		printf("Hauptmenue\n");
		printf("==========\n");
		printf("0. Ende\n");
		printf("1. Einstellungen\n");
		printf("2. Eprom leer?\n");
		printf("3. Eprom -> Datei\n");
		printf("4. Eprom mit Datei vergleichen\n");
		printf("5. Datei -> Eprom (brennen)\n");
		printf("6. Debugging/Analyse des Eprommers\n");
		printf("7. 16-Bit Datei in 8-Bit Dateien splitten\n");
		printf("8. 8-Bit Dateien zu 16-Bit Datei zusammenfassen\n");
		printf("9. Copyright und Gewaehrleistung\n");

		i=GetNum();
		if      (i==1) Einstellungen();
		else if (i==2) Testen();
		else if (i==3) Auslesen();
		else if (i==4) Vergleichen();
		else if (i==5) Brennen();
		else if (i==6) debug();
		else if (i==7) split();
		else if (i==8) join();
		else if (i==9) Copyright();
	
		}
}



/*----------------------------------------------------------*/
/* main                                                     */
/*----------------------------------------------------------*/

char dummy[80];


void Init(){
	U3OE=U3OE_OFF;
	WriteCommand(0);
	cd4094Init(8,&u3);
	cd4094Init(4,&u4);
	cd4094Init(2,&u5);
	Reset4040();
	act=0;
	/* 4021: egal (wird bei jedem strobe fuer 4094 resettet...) */

	/* kleine lichtorgel zur begruessung (SCNR...) */
	LED(TRUE);usleep(120000);	LED(FALSE);usleep(120000);
	LED(TRUE);usleep(120000);	LED(FALSE);usleep(120000);
	LED(TRUE);usleep(120000);	LED(FALSE);usleep(120000);
	LED(TRUE);usleep(480000);	LED(FALSE);
}


void Done(){
	U3OE=U3OE_OFF;
	WriteCommand(0);
	cd4094Write(&u3,0);
	cd4094Write(&u4,0);
	cd4094Write(&u5,0);
}


main(int argc, char *argv[]){
int err;

	Optionen.port=LPT_DEFAULT;	/* ein defaultwert muss hier bereits gesetzt sein! */

	if (ioperm(LPTBASE,3,1)) {
		err=errno;
		fprintf(stderr,"ioperm($%lx,3,1) failed (%d): %s\n",LPTBASE,err,strerror(err));
		if (err==1) fprintf(stderr,"(this program requires root privilges)\n");
		return(1);
		}

	if ((tio_id=open("/dev/tio"))<0) {
		err=errno;
		fprintf(stderr,"cant open tio device: %s!\n",strerror(err));
		fprintf(stderr,
			"This program requires the 'tio.o' loadable kernel module\n"
			"(e.g. with 'insmod tio.o') and the device file /dev/tio\n"
			"(e.g. 'make device' or 'mknod /dev/tio c 60 128').\n"
			"Read the included docs for more informations.\n");
	
		ioperm(LPTBASE,3,0);
		return(1);
		}

	Init();
	Hauptmenue();
	Done();

	close(tio_id);
	ioperm(LPTBASE,3,0);
	return(0);
}


/*----------------------------------------------------------*/
/* end of file                                              */
/*----------------------------------------------------------*/
